<?php

class ITSEC_Malware_Scheduling {

	// Approximately 10 hours since first try
	const MAX_TRIES = 8;

	public function run() {
		add_action( 'itsec_malware_scheduled_scan', array( $this, 'run_scan' ) );
		add_action( 'itsec_malware_retry_scheduled_scan', array( $this, 'retry_scan' ) );

		add_action( 'ithemes_sync_register_verbs', array( $this, 'register_sync_verbs' ) );
		add_filter( 'itsec-filter-itsec-get-everything-verbs', array( $this, 'register_sync_get_everything_verbs' ) );
		add_action( 'itsec_change_admin_user_id', array( $this, 'update_notification_user_id_on_admin_change' ) );
	}

	public function run_scan() {

		// Don't retry twice daily scan if a retry is still scheduled due to the site missing cron executions.
		if ( $this->is_retry_scheduled() ) {
			return;
		}

		require_once( dirname( __FILE__ ) . '/class-itsec-malware-scheduling-scanner.php' );

		$results = ITSEC_Malware_Scheduling_Scanner::scan();

		if ( is_wp_error( $results ) && $results->get_error_message( 'itsec-scheduled-malware-scanner-sucuri-error' ) ) {
			$this->schedule_retry(1 );
		}
	}

	public function retry_scan( $try = 1 ) {
		require_once( dirname( __FILE__ ) . '/class-itsec-malware-scheduling-scanner.php' );

		$results = ITSEC_Malware_Scheduling_Scanner::scan();

		if ( ! is_wp_error( $results ) || ! $results->get_error_message( 'itsec-scheduled-malware-scanner-sucuri-error' ) ) {
			return;
		}

		$next_try = $try + 1;

		if ( $next_try > self::MAX_TRIES ) {
			ITSEC_Malware_Scheduling_Scanner::send_email( $results );
		} elseif ( ! $this->is_retry_scheduled() ) {
			$this->schedule_retry( $next_try );
		}
	}

	private function schedule_retry( $retry_count ) {

		$when = time() + ( 60 * $this->minutes_until_retry( $retry_count ) );

		wp_schedule_single_event( $when, 'itsec_malware_retry_scheduled_scan', array( $retry_count ) );
	}

	private function is_retry_scheduled() {
		$crons = _get_cron_array();

		if ( empty( $crons ) ) {
			return false;
		}

		foreach ( $crons as $timestamp => $cron ) {
			if ( isset( $cron['itsec_malware_retry_scheduled_scan'] ) ) {
				return true;
			}
		}

		return false;
	}


	public static function update_schedule() {
		$hook = 'itsec_malware_scheduled_scan';

		// Unschedule any existing schedules. This is to prevent faults from creating multiple schedules.
		$timestamp = wp_next_scheduled( $hook );
		$count = 0;

		while ( false !== $timestamp ) {
			wp_unschedule_event( $timestamp, $hook );

			$timestamp = wp_next_scheduled( $hook );

			if ( ++$count > 20 ) {
				// Prevent endless loops.
				break;
			}
		}

		wp_schedule_event( time(), 'twicedaily', $hook );
	}


	/**
	 * Determine the number of minutes we should wait before contacting Sucuri.
	 *
	 * Exponential back-off modified to have a minimum of 10 minute difference.
	 *
	 * [ ( 2ˣ-1 ) / 2 ] + 10x
	 *
	 * @param int $try Try count.
	 *
	 * @return int
	 */
	private function minutes_until_retry( $try ) {
		return (int) floor( ( ( pow(2, $try ) - 1 ) / 2 ) + ( 10 * $try ) );
	}

	/**
	 * Register verbs for Sync.
	 *
	 * @since 3.6.0
	 *
	 * @param Ithemes_Sync_API $api Sync API object.
	 */
	public function register_sync_verbs( $api ) {
		$api->register( 'itsec-get-malware-schedule-settings', 'Ithemes_Sync_Verb_ITSEC_Get_Malware_Schedule_Settings', dirname( __FILE__ ) . '/sync-verbs/itsec-get-malware-schedule-settings.php' );
		$api->register( 'itsec-manage-malware-schedule-settings', 'Ithemes_Sync_Verb_ITSEC_Manage_Malware_Schedule_Settings', dirname( __FILE__ ) . '/sync-verbs/itsec-manage-malware-schedule-settings.php' );
	}

	/**
	 * Filter to add verbs to the response for the itsec-get-everything verb.
	 *
	 * @since 3.6.0
	 *
	 * @param array $verbs
	 *
	 * @return array Array of verbs.
	 */
	public function register_sync_get_everything_verbs( $verbs ) {
		$verbs['malware_scheduling'][] = 'itsec-get-malware-schedule-settings';

		return $verbs;
	}

	/**
	 * Update the notification settings when the admin user id changes.
	 *
	 * @since 4.1.0
	 *
	 * @param int $new_user_id
	 */
	public function update_notification_user_id_on_admin_change( $new_user_id ) {

		$settings = ITSEC_Modules::get_settings_obj( 'malware-scheduling' );
		$contacts = $settings->get( 'email_contacts' );

		if ( ! is_array( $contacts ) ) {
			return;
		}

		$changed = false;

		foreach ( $contacts as $i => $contact ) {
			if ( is_numeric( $contact ) && 1 === (int) $contact ) {
				$contacts[ $i ] = $new_user_id;
				$changed = true;
				break;
			}
		}

		if ( $changed ) {
			$settings->set( 'email_contacts', $contacts );
		}
	}
}
